package ags.communication;
import gnu.io.CommPort;
import gnu.io.SerialPort;
import gnu.io.UnsupportedCommOperationException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
/**
 * The generic host represents a common body of methods that can be used to
 * build any sort of communication program, specifically one that is capable of
 * interfacing with an apple computer at the standard applesoft basic prompt.
 *
 * Methods for error-resistant data transmission and advanced scripting support
 * are also provided for future reuse.
 * 
 */
public class GenericHost {
    
    /**
     * Enumeration of various flow control modes
     */
    public enum FlowControl {
        /**
         * No flow control -- ignore hardware flow control signals
         */
        none(SerialPort.FLOWCONTROL_NONE),
        /**
         * XON/XOFF flow control mode
         */
        xon(SerialPort.FLOWCONTROL_XONXOFF_IN),
        /**
         * Hardware flow control
         */
        hardware(SerialPort.FLOWCONTROL_RTSCTS_IN);
  
        /**
         * Value of flow control mode used by RXTX internally
         */
        private int val;
        /**
         * Constructor for enumeration
         * @param v RXTX constant value
         */
        FlowControl(int v) {
            val = v;
        }
        /**
         * Get RXTX constant for this flow control mode
         * @return desired flow control mode value
         */
        public int getConfigValue() {
            return val;
        }
    }
    
    /**
     * Active com port
     */
    private SerialPort port = null;

    /**
     * input stream for com port
     */
    private InputStream in = null;
    /**
     * output stream for com port
     */
    private OutputStream out = null;
    /**
     * Most recently set baud rate
     */
    int currentBaud = 1;
    /**
     * Current flow control mode being used
     */
    FlowControl currentFlow = FlowControl.none;
    
    /**
     * The familiar monitor prompt: *
     */
    public static final String MONITOR_PROMPT = "*";
    /**
     * The familiar applesoft prompt: ]
     */
    public static final String APPLESOFT_PROMPT = "]";
    /**
     * CTRL-X is used to cancel a line of input
     */
    public static char CANCEL_INPUT = 18; // CTRL-X cancels the input
    /**
     * Do we expect sent characters to echo back to us?
     */
    boolean expectEcho;
    
    /**
     * Creates a new instance of GenericHost
     * @param port Open com port ready to use
     */
    public GenericHost(CommPort port) {
        expectEcho = true;
//        expectEcho = false;
        this.port = (SerialPort) port;
        try {
            in = port.getInputStream();
            out = port.getOutputStream();
            this.port.setDTR(true);
        } catch(IOException ex) {
            ex.printStackTrace();
        }
    }
    
    /**
     * Configure the port to the desired baud rate, 8-N-1
     * @param baudRate Baud rate (e.g. 300, 1200, ...)
     */
    public void setBaud(int baudRate) {
        try {
            port.setSerialPortParams(baudRate, port.DATABITS_8, port.STOPBITS_1, port.PARITY_NONE);
            port.setFlowControlMode(currentFlow.val);
            currentBaud = baudRate;
        } catch(UnsupportedCommOperationException ex) {
            ex.printStackTrace();
        }
    }
    
    /**
     * Given a file name, the contents of that file are sent verbatim to the apple.
     * After every line, the monitor prompt (*) is expected in order to continue.
     * @param executeHexFile name of file
     * @throws java.io.IOException if the file or com port cannot be accessed
     */
    public void executeScript(String executeHexFile)
    throws IOException {
        String driverData = DataUtil.getFileAsString(executeHexFile);
        String lines[] = driverData.split("\n");
        String searchForLabel = null;
        String errorLabel = null;
        for(int i = 0; i < lines.length; i++) {
            boolean fatalError = false;
            String printLine = lines[i] + "\r";
//            System.out.println(printLine);
            if (searchForLabel != null) {
                if (lines[i].toLowerCase().equals(":"+searchForLabel.toLowerCase())) {
                    System.out.println("Skipping to label "+searchForLabel);
                    searchForLabel = null;
                }
                continue;
            }
            try {
                if (lines[i].startsWith(";")) continue;
                else if (lines[i].startsWith("!")) {
                    String[] command = printLine.substring(1).toLowerCase().split("\\s");
                    String cmd = command[0];
                    if (cmd.equals("goto")) {
                        searchForLabel = command[1];
                    } else if (cmd.equals("onerror")) {
                        errorLabel = command[1];
                    } else if (cmd.equals("baud")) {
                        int baud = Integer.parseInt(command[1]);
                        setBaud(baud);
                        cancelLine();
                    } else if (cmd.equals("flow")) {
                        currentFlow = FlowControl.valueOf(command[1]);
                        setBaud(currentBaud);
                    } else if (cmd.equals("error")) {
                        fatalError = true;
                        throw new IOException(printLine.substring(1));
                    } else if (cmd.equals("wait")) {
                        int waitTime = Integer.parseInt(command[1]);
                        DataUtil.wait(waitTime);
                    } else if (cmd.equals("expect")) {
                        String v = command[1].toLowerCase();
                        boolean val = v.equals("true") || v.equals("on") || v.equals("1");
                        expectEcho = val;
                        System.out.println("Expect echo: "+String.valueOf(val));
                    } else {
                        System.out.println("Unknown command: "+command[0]);
                    }
                } else if (lines[i].startsWith(MONITOR_PROMPT)) {
                    writeParanoid(printLine.substring(1));
                    expectMonitor();
                } else if (lines[i].startsWith(APPLESOFT_PROMPT)) {
                    writeParanoid(printLine.substring(1));
                    expectApplesoft();
                } else if (lines[i].startsWith("~")) {
                    write(printLine.substring(1));
                } else if (lines[i].startsWith("'")) {
                    System.out.println(printLine.substring(1));
                } else if (lines[i].startsWith("\"")) {
                    writeParanoid("480:"+DataUtil.asAppleScreenHex(printLine.substring(1))+"\r");
                    expectMonitor();
                } else {
                    writeParanoid(printLine);
                }
            } catch (NumberFormatException ex) {
                if (fatalError) throw ex;
                ex.printStackTrace();
                System.out.println("You should check the script for bad numerical information (e.g. baud rate settings)");
                searchForLabel = errorLabel;
                errorLabel = null;
            } catch (IOException ex) {
                System.out.println("error in script, line "+(i+1)+":"+ex.getMessage());
                if (errorLabel == null || fatalError) throw ex;
                searchForLabel = errorLabel;
                errorLabel = null;
            }
        }
    }
    
    /**
     * Write a string to the remote host.
     * If expectEcho == true, then data sent is also expected to echo back to us.  If the data is not echoed back in a reasonable amount of time, the command to cancel the line is sent and the process is retried a few times.
     * If expectEcho == false, the data is sent blindly and a calculated amount of time is ellapsed before continuing on.
     * @param s String to send
     * @throws java.io.IOException If the line could not be written
     */
    public void writeParanoid(String s) throws IOException {
        byte bytes[] = s.getBytes();
        byte[] expect = new byte[1];
        int errors = 0;
        for (int i=0; i < bytes.length && errors < 2; i++) {
            //System.out.println(i+"="+bytes[i]);
            try {
                writeOutput(bytes, i, 1);
                if (bytes[i] >= 32 && expectEcho) {
                    expect(s.substring(i,i+1), 500, false);
                } else {
                    DataUtil.wait(5);
                }
                errors=0;
            } catch (IOException ex) {
                System.out.println("Failure writing line, retrying...");
                cancelLine();      // control-x
                i=-1;
                errors++;
//                ex.printStackTrace();
            }
        }
        if (errors >= 2) throw new IOException("Cannot write "+s);
    }

    /**
     * Sends a CTRL-X character, which cancels the current line on the remote host
     * @throws java.io.IOException If there is trouble sending data to the remote host
     */
    protected void cancelLine() throws IOException {
        writeOutput(new byte[]{(byte) CANCEL_INPUT});      // control-x
    }
    
    /**
     * Does this byte buffer contain the same values as the array of bytes?
     * @param bb buffer to check
     * @param data bytes to look for
     * @return true if bytes were found in the buffer
     */
    public static boolean bufferContains(ByteBuffer bb, byte data[]) {
        int d = 0;
        boolean match = false;
        for(int i = 0; i < bb.position() && d < data.length; i++)
            if(bb.get(i) == data[d]) {
            match = true;
            d++;
            } else {
            match = false;
            d = 0;
            }
        
        return match;
    }
    
    /**
     * Expect the monitor prompt to be sent back (if expectEcho == true)
     * @throws java.io.IOException If the monitor prompt is not sent to us
     */
    public void expectMonitor() throws IOException {
        if (expectEcho) expect(MONITOR_PROMPT, 5000, false);
        else DataUtil.wait(500);
    }
    
    /**
     * Expect the applesoft prompt to be sent back (if expectEcho == true)
     * @throws java.io.IOException If the applesoft prompt is not sent to us
     */
    public void expectApplesoft() throws IOException {
        if (expectEcho) expect(APPLESOFT_PROMPT, 5000, false);
        else DataUtil.wait(500);
    }
    
    /**
     * Expect a specific set of bytes to be sent in a given amount of time
     * @param data data expected
     * @param timeout max time to wait for data
     * @throws java.io.IOException if there is a timeout waiting for the data
     * @return true
     */
    public boolean expectBytes(byte data[], int timeout)
    throws IOException {
        ByteBuffer bb = ByteBuffer.allocate(Math.max(80,Math.max(inputAvailable(),data.length * 2)));
        while(timeout > 0) {
            for(; inputAvailable() == 0 && timeout > 0; timeout -= 5)
                DataUtil.wait(5);
            
            byte receivedData[] = getBytes();
            bb.put(receivedData);
            if(bufferContains(bb, data))
                return true;
        }
        if(bb.position() == 0)
            throw new IOException((new StringBuilder()).append("expected ").append(Arrays.toString(data)).append(" but timed out").toString());
        else
            throw new IOException((new StringBuilder()).append("Expected ").append(Arrays.toString(data)).append(" but got ").append(Arrays.toString(bb.array())).toString());
    }
    
    /**
     * Expect a specific string to be sent in a given amount of time
     * @param string data expected
     * @param timeout max time to wait for data
     * @param noConversion if true, string is treated as-is.  If false, expected data is converted to the high-order format that the apple sends back natively
     * @return true
     * @throws java.io.IOException if there is a timeout waiting for the data
     */
    public boolean expect(String string, int timeout, boolean noConversion)
    throws IOException {
        String searchString = "";
        while(timeout > 0) {
            for(; inputAvailable() == 0 && timeout > 0; timeout -= 10)
                DataUtil.wait(10);
            
            String receivedString = getString();
            if(!noConversion)
                receivedString = DataUtil.convertFromAppleText(receivedString);
            searchString = (new StringBuilder()).append(searchString).append(receivedString).toString();
            if(searchString.contains(string))
                return true;
        }
        if(searchString.equals(""))
            throw new IOException((new StringBuilder()).append("expected ").append(string).append(" but timed out").toString());
        else
            throw new IOException((new StringBuilder()).append("Expected ").append(string).append(" but got ").append(searchString).toString());
    }
    
    /**
     * Get all avail. com input data as a string
     * @throws java.io.IOException if there is a problem with the port
     * @return string of data
     */
    public String getString()
    throws IOException {
        if(inputAvailable() == 0) {
            return "";
        } else {
            byte data[] = new byte[inputAvailable()];
            readInput(data);
            return new String(data);
        }
    }
    
    /**
     * Get all avail. com input data as an array of bytes
     * @throws java.io.IOException if there is a problem with the port
     * @return data
     */
    public byte[] getBytes()
    throws IOException {
        byte data[] = new byte[inputAvailable()];
        readInput(data);
        return data;
    }
    
    /**
     * write a string out and sleep a little to let the buffer empty out
     * @param s String to write out
     * @throws java.io.IOException If the data could not be sent
     */
    public void write(String s) throws IOException {
        byte bytes[] = s.getBytes();
        int blockSize = (currentBaud == 300 ? 80 : 2);
//            if(currentBaud == 300) waitTime += 500;
        for (int i=0; i < bytes.length; i+=blockSize) {
            int toGo = Math.min(blockSize, bytes.length - i);
            int waitTime = Math.max(10, (toGo * 10000) / currentBaud);
            Date d1 = new Date();
            writeOutput(bytes, i, toGo);
            Date d2 = new Date();
            int diff = (int)(d2.getTime() - d1.getTime());
            if(diff < waitTime) DataUtil.wait(waitTime - diff);
        }
    }
    
    //-------------------------------------
    //--- Raw i/o routines, isolated for better debugging and flow control support
    //-------------------------------------
    
    /**
     * Returns the number of bytes available in the input buffer
     * @throws java.io.IOException If the port cannot be accessed
     * @return Number of available bytes of input in buffer
     */
    protected int inputAvailable() throws IOException {
//        System.out.println("inputAvailable - start");
        int avail = 0;
        startReadOperation();
//        System.out.println("inputAvailable - reading available");
        avail = in.available();
        endReadOperation();
//        System.out.println("inputAvailable = "+avail);
        return avail;
    }
    
    private void startReadOperation() {
//        System.out.println("Starting read operation");
        if (currentFlow != FlowControl.none) {
//            System.out.println("Setting RTS");
            if (currentFlow == FlowControl.hardware) port.setRTS(true);
            try {
//                System.out.println("Setting receive timeout");
                port.enableReceiveTimeout(500);
//                System.out.println("Setting receive threshold");
                port.enableReceiveThreshold(1);
            } catch (UnsupportedCommOperationException ex) {
                ex.printStackTrace();
            }
        }        
//        System.out.println("Ending read operation");
    }
    
    private void endReadOperation() {
        if (currentFlow != FlowControl.none) {
//            System.out.println("Clearing RTS");
            if (currentFlow == FlowControl.hardware) port.setRTS(false);
//            System.out.println("Disabling receive threshold");
            port.disableReceiveThreshold();
//            System.out.println("Disabling receive timeout");
            port.disableReceiveTimeout();
        }        
    }
    
    /**
     * Fill the provided byte[] array with data if possible
     * @param buffer buffer to fill
     * @throws java.io.IOException If data could not be read for any reason
     * @return Number of bytes read into buffer
     */
    protected int readInput(byte[] buffer) throws IOException {
        startReadOperation();
//        System.out.println("Reading data");
        int size = in.read(buffer);
//        System.out.println("Read "+size+" bytes");
        endReadOperation();
        return size;        
    }
    
    /**
     * Write entire buffer to remote host
     * @param buffer Buffer of data to send
     * @throws java.io.IOException If data could not be sent
     */
    protected void writeOutput(byte[] buffer) throws IOException {
        writeOutput(buffer, 0, buffer.length);
    }
    
    /**
     * Wait x number of milliseconds for remote host to allow us to send data (if flow control == hardware mode)
     * @param timeout Number of milliseconds to wait before throwing timeout error
     * @throws java.io.IOException If we timed out waiting for "clear to send" signal
     */
    protected void waitToSend(int timeout) throws IOException {
        if (currentFlow == FlowControl.hardware) {
//            System.out.println("Waiting for CTS");
            while (!port.isCTS() && timeout > 0) {
                DataUtil.wait(10);
                timeout -= 10;
            }
            if (timeout <= 0) throw new IOException("Timed out waiting to send data to remote host!");
//            System.out.println("Finished waiting for CTS");
        }
    }
    
    /**
     * Write data to host from provided buffer
     * @param buffer Buffer of data to send
     * @param offset Starting offset in buffer to write
     * @param length Length of data to write
     * @throws java.io.IOException If there was trouble writing to the port
     */
    protected void writeOutput(byte[] buffer, int offset, int length) throws IOException {
        if (buffer == null || offset >= buffer.length || buffer.length == 0 || length == 0) return;
//        System.out.println("Sending "+length+" bytes");
        if (currentFlow == FlowControl.hardware) {
            for (int i=offset; i < offset+length; i++) {
                waitToSend(2000);   // Really, 2 seconds should be sufficient!
                out.write(buffer, i, 1);
            }
        } else {
            out.write(buffer, offset, length);        
        }
//        System.out.println("Finished sending data");
    }
}